/*************************************************************************************************
 *
 *   Copyright (c) Hilscher GmbH. All Rights Reserved.
 *
 *************************************************************************************************
 * @file CifX_ApDemo.cpp
 *
 * @brief This is the base file for all Hilscher CifX RTE example applications.
 * It contains the main entry function.
 * 
 * @author Elmar & Julien
 *
 */

/* default stuff */
#include "stdafx.h"

/* own header */
#include "CifX_ApDemo.h"

/* application header */
#include "Application.h"


/*************************************************************************************************
 * @brief This method is the main entry point for the application.
 * 
 * @param argc Number of arguments.
 * @param argv Poniter to arguments.
 *
 * @return int Return 0 (TLR_S_OK).
 *
 */
int _tmain(int argc, _TCHAR* argv[])
{
  /* return value of main function */
  TLR_RESULT tResult = TLR_S_OK;
  
  /* name of board to use */
  char* szBoardName = "CIFx0";

  /* number of channel to use */
  TLR_UINT32 ulChannel = 0;

  /* general reference to driver */
  CIFXHANDLE  hDriver  = NULL;

  /* general reference to communication channel */
  CIFXHANDLE  hChannel = NULL;

  /* common variables for packets */
  CIFX_PACKET tSendPkt  = {{0}};
  CIFX_PACKET tRecvPkt  = {{0}};
  CIFX_PACKET* ptRecvPkt = NULL;
  
  /* pointer to common application data */
  APPLICATION_T* ptApp = {{0}};

  /* variable for host state */
  TLR_UINT32  ulState     = 0;

  /* default timeout for driver api calls */
  TLR_UINT32  ulTimeout   = 100;        

  DEBUG("**********************************************************\n");
  DEBUG("*                                                        *\n");
  DEBUG("*   Basic sample application for Ethernet POWERLINK.     *\n");
  DEBUG("*                                                        *\n");
  DEBUG("*   Copyright (c) Hilscher GmbH. All Rights Reserved.    *\n");
  DEBUG("*                                                        *\n");
  DEBUG("**********************************************************\n");
  DEBUG("\n");

  /* Open Driver */
  DEBUG("Opening driver...\n");
  tResult = xDriverOpen(&hDriver);
  
  if (CIFX_NO_ERROR == tResult)
  {
    /* Driver successfully opened */
    /* Open channel */
    DEBUG("Opening channel %d on board %s...\n", ulChannel, szBoardName);
    tResult = xChannelOpen(hDriver, szBoardName, ulChannel, &hChannel);

    if (CIFX_NO_ERROR == tResult)
    {
      /* start hardware with new configuration */
      DEBUG("Processing system restart...\n");
      tResult = xChannelReset(hChannel, CIFX_SYSTEMSTART, 2000);
      
      /* system restart successful */
      if (CIFX_NO_ERROR == tResult)
      {
        /* Toggle Application Ready State Flag */
        do
        {
          tResult = xChannelHostState(hChannel, CIFX_HOST_STATE_READY, &ulState, ulTimeout);
          
          /* if Dev is not ready, retry that action */
          if (CIFX_DEV_NOT_READY == tResult)
          {
            /* retry after 500 ms */
            Sleep(500);
          }
        } 
        while (tResult == CIFX_DEV_NOT_READY);      


        /* check for CIFX_NO_ERROR added because return value  of xChannelHostState() changed */
        /* between SHM-lib <=0.930 and >=0.940 */
        if ((CIFX_NO_ERROR == tResult) || (CIFX_DEV_NOT_RUNNING == tResult))
        {
          /* build configuration packet */
          tResult = BuildConfigurationReq(&tSendPkt);

          /* send configuration Req packet */
          DEBUG("Sending configuration request...\n");
          tResult = xChannelPutPacket(hChannel, &tSendPkt, ulTimeout);

          /* get configuration Cnf packet */
          tResult = xChannelGetPacket(hChannel, sizeof(tRecvPkt), &tRecvPkt, ulTimeout);

          /* check if we got an error within configuration packet */
          if (TLR_S_OK != tResult)
          {
            DEBUG("Configuration Confirmation not received.");
          }
          else if (TLR_S_OK != tRecvPkt.tHeader.ulState)
          {
            DEBUG("Configuration packet returned with error code: 0x%x\n", tRecvPkt.tHeader.ulState );
          }
          else
          {
            /* start hardware with new configuration */
            DEBUG("Processing channel init...\n");
            tResult = xChannelReset(hChannel, CIFX_CHANNELINIT, 2000);
          
            if (CIFX_NO_ERROR == tResult)
            {
              /* Waiting for netX warmstarting */
              do
              {
                tResult = xChannelHostState(hChannel, CIFX_HOST_STATE_READY, &ulState, ulTimeout);
              }
              while (CIFX_DEV_NOT_RUNNING == tResult);
            
            
              /* check CifX state */
              if ((CIFX_NO_ERROR == tResult) || (CIFX_DEV_NO_COM_FLAG == tResult))
              {
                /* initialize the application */
                tResult = App_Initialize(&ptApp, hDriver, hChannel, ulTimeout);

				        /* check if initializing the application succeeded */
				        if ((TLR_S_OK == tResult) && (NULL != ptApp))
				        {
					        /* bus on */
					        DEBUG("Setting bus state on...\n");
					        tResult = xChannelBusState(hChannel, CIFX_BUS_STATE_ON, &ulState, ulTimeout);        
        	                
					        /* bus activity begins here */
					        if (CIFX_BUS_STATE_ON == ulState)
					        {
					          DEBUG("Entering endless loop...\n");
					          DEBUG("\n");
					          DEBUG(">> Press any key to leave. <<\n");
					          DEBUG("\n");

					          /* set return to TLR_S_OK to initially enter the loop */
					          tResult = TLR_S_OK;
					          while(TLR_S_OK == tResult || CIFX_DEV_GET_NO_PACKET == tResult)
					          {
						          /* do process stuff first */
						          /* handling of io depends on the bus system,  */
						          /* there might be some bits we have to toggle */
						          HandleProcessData(ptApp);                   

						          /* now we do the acyclic stuff*/
						          tResult = xChannelGetPacket(hChannel, sizeof(tRecvPkt), &tRecvPkt, 5);

						          /* check if we have really received a packet or an error code */
						          if(TLR_S_OK == tResult)
						          {
						            /* signalize if we got a packet*/
						            DEBUG("-");

						            /* handle packets */
						            HandlePacket(ptApp, &tRecvPkt);
						          }

	            #ifdef WIN32
						          /* enable leaving of endless loop if running under windows, */
						          /* necessary to return handles and close driver */
						          if (_kbhit())
						          {
						            tResult = TLR_E_FAIL;
						          }
	            #endif
          	                    
					          } /* while(TLR_S_OK == tResult || CIFX_DEV_GET_NO_PACKET == tResult) --> packet loop */

					          /* force at least one empty line within console output */
					          DEBUG("\n\n");

					        } /* Bus on */
        	            
					        /* Finalize application in order to dispose handles, free memory, aso.*/
					        App_Finalize(ptApp);

				        } /* check initialize application succeeded */

              } /* check CifX state */

            } /* cifX channel init */

          } /* check configuration packet */

        } /* check for CIFX_NO_ERROR added because return value of xChannelHostState() changed */

      } /* cifX successfully reset */
      
      DEBUG("Closing Channel...\n");
      tResult = xChannelClose(hChannel);
    } /* channel successfully opened */

    DEBUG("Closing Driver...\n");
    tResult = xDriverClose(hDriver);

  } /* driver successfully opened */

  return tResult;
}


